program RungeKF45;
{--------------------------------------------------------------------}
{  Alg9'5.pas   Pascal program for implementing Algorithm 9.5        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 9.5 (Runge-Kutta-Fehlberg Method RKF45).                }
{  Section   9.5, Runge-Kutta Methods, Page 461                      }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    TolLo = 5E-8;
    TolHi = 1E-5;
    Tol = 2E-5;
    GNmax = 630;
    MaxM = 630;
    FunMax = 9;

  type
    VECTOR = array[0..MaxM] of real;
    LETTERS = string[200];
    States = (Changes, Done, Working);
    DoSome = (Go, Stop);

  var
    FunType, GNpts, Inum, M, Mend, Step, Sub: integer;
    A, B, Hstart, Rnum, Y0: real;
    Ans: CHAR;
    T, X, Y: VECTOR;
    Mess: LETTERS;
    State: States;
    DoMo: DoSome;

  function F (T, Y: real): real;
  begin
    case FunType of
      1: 
        F := (T - Y) / 2;
      2: 
        F := (Y - T) / 2;
      3: 
        F := T * T - Y;
      4: 
        F := 3 * T + 3 * Y;
      5: 
        F := -T * Y;
      6: 
        F := EXP(-2 * T) - 2 * Y;
      7: 
        F := 2 * T * Y * Y;
      8: 
        F := 1 + Y * Y;
      9: 
        F := T * T + Y * Y;
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      1:
        WRITELN('Y` = (T-Y)/2');
      2: 
        WRITELN('Y` = (Y-T)/2');
      3: 
        WRITELN('Y` = T^2 - Y');
      4: 
        WRITELN('Y` = 3*T + 3*Y');
      5: 
        WRITELN('Y` = -T*Y');
      6: 
        WRITELN('Y` = EXP(-2*T) - 2*Y');
      7:
        WRITELN('Y` = 2*T*Y^2');
      8:
        WRITELN('Y` = 1 + Y^2');
      9: 
        WRITELN('Y` = T^2 + Y^2');
    end;
  end;

  procedure RKF45 (A, B, Y0: real; Tol, TolLo, TolHi: real;
                   Step: integer; var T, Y: VECTOR; var M, Mend: integer);
    label
      999;
    const
      Big = 1E13;
    type
      SVECTOR = array[0..30] of real;
    var
      HC, I, J, K: integer;
      Br, Delta, H, Hmax, Hmin, Err, K1, K2, K3, K4, K5, K6, TJ, YJ: real;
      A2, B2, A3, B3, C3, A4, B4, C4, D4, A5, B5, C5, D5, E5, S, Small: real;
      A6, B6, C6, D6, E6, F6, R1, R3, R4, R5, R6, N1, N3, N4, N5: real;
      M1, M3, M4, M5, M6, RelErr, Y1, Y2, Y3, Y4, Y5, Y6, Ygood, Ynew: real;
      H0: SVECTOR;
    procedure CONSTANTS;
    begin
      A2 := 1 / 4;
      B2 := 1 / 4;
      A3 := 3 / 8;
      B3 := 3 / 32;
      C3 := 9 / 32;
      A4 := 12 / 13;
      B4 := 1932 / 2197;
      C4 := -7200 / 2197;
      D4 := 7296 / 2197;
      A5 := 1;
      B5 := 439 / 216;
      C5 := -8;
      D5 := 3680 / 513;
      E5 := -845 / 4104;
      A6 := 1 / 2;
      B6 := -8 / 27;
      C6 := 2;
      D6 := -3544 / 2565;
      E6 := 1859 / 4104;
      F6 := -11 / 40;
      R1 := 1 / 360;
      R3 := -128 / 4275;
      R4 := -2197 / 75240.0;
      R5 := 1 / 50;
      R6 := 2 / 55;
      N1 := 25 / 216;
      N3 := 1408 / 2565;
      N4 := 2197 / 4104;
      N5 := -1 / 5;
      M1 := 16 / 135;
      M3 := 6656 / 12825;
      M4 := 28561 / 56430.0;
      M5 := -9 / 50;
      M6 := 2 / 55;
      H0[0] := 1;
      H0[1] := 5E-1;
      H0[2] := 2E-1;
      H0[3] := 1E-1;
      H0[4] := 5E-2;
      H0[5] := 2E-2;
      H0[6] := 1E-2;
      H0[7] := 5E-3;
      H0[8] := 2E-3;
      H0[9] := 1E-3;
      H0[10] := 5E-4;
      H0[11] := 2E-4;
      H0[12] := 1E-4;
      H0[13] := 5E-5;
      H0[14] := 2E-5;
      H0[15] := 1E-5;
      H0[16] := 5E-6;
      H0[17] := 2E-6;
      H0[18] := 1E-6;
      H0[19] := 5E-7;
      H0[20] := 2E-7;
      H0[21] := 1E-7;
    end;
  begin
    CONSTANTS;
    if Step = 2 then
      for I := 1 to 21 do
        H0[I] := H0[I - 1] / 2;
    HC := 0;
    for I := 0 to 17 do
      if (H0[I] <= Hstart) and (HC = 0) then
        HC := I;
    for I := 0 to 17 do
      if (ABS(B - A) < H0[I]) then
        HC := I;
    H := H0[HC];
    Hmax := 1;
    Hmin := 2E-6;
    Small := TolLo;
    T[0] := A;
    Y[0] := Y0;
    T[0] := A;
    J := 0;
    TJ := A;
    Br := B - Hmin;
    while T[J] < B do
      begin
        if T[J] + H > Br then
          H := B - T[J];
        TJ := T[J];
        YJ := Y[J];
        Y1 := YJ;
        K1 := H * F(TJ, Y1);
        Y2 := YJ + B2 * K1;
        if Big < ABS(Y2) then
          goto 999;
        K2 := H * F(TJ + A2 * H, Y2);
        Y3 := YJ + B3 * K1 + C3 * K2;
        if Big < ABS(Y3) then
          goto 999;
        K3 := H * F(TJ + A3 * H, Y3);
        Y4 := YJ + B4 * K1 + C4 * K2 + D4 * K3;
        if Big < ABS(Y4) then
          goto 999;
        K4 := H * F(TJ + A4 * H, Y4);
        Y5 := YJ + B5 * K1 + C5 * K2 + D5 * K3 + E5 * K4;
        if Big < ABS(Y5) then
          goto 999;
        K5 := H * F(TJ + A5 * H, Y5);
        Y6 := YJ + B6 * K1 + C6 * K2 + D6 * K3 + E6 * K4 + F6 * K5;
        if Big < ABS(Y6) then
          goto 999;
        K6 := H * F(TJ + A6 * H, Y6);
        Err := ABS(R1 * K1 + R3 * K3 + R4 * K4 + R5 * K5 + R6 * K6);
        Ynew := YJ + N1 * K1 + N3 * K3 + N4 * K4 + N5 * K5;
        Ygood := YJ + M1 * K1 + M3 * K3 + M4 * K4 + M5 * K5 + M6 * K6;
        Err := ABS(Ygood - Ynew);
        RelErr := Err / (ABS(Ynew) + ABS(Ygood) + Small);
        if (Step = 1) or (Step = 2) then
          begin
            if (RelErr < Tolhi) or (17 <= HC) then
              begin
                Y[J + 1] := Ynew;
                if TJ + H > Br then
                  T[J + 1] := B
                else
                  T[J + 1] := TJ + H;
                J := J + 1;
                TJ := T[J];
              end;
            if (RelErr >= Tolhi) and (HC < 17) then
              begin
                HC := HC + 1;
                H := H0[HC];
              end;
            if (RelErr <= TolLo) and (HC > 0) then
              begin
                HC := HC - 1;
                H := H0[HC];
              end;
          end;
        if Step = 3 then
          begin
            if (Err < Tol) or (H <= 2 * Hmin) then
              begin
                Y[J + 1] := Ynew;
                if TJ + H > Br then
                  T[J + 1] := B
                else
                  T[J + 1] := TJ + H;
                J := J + 1;
                TJ := T[J];
              end
            else
              begin
                Delta := 0.84 * SQRT(SQRT(Tol / Err));
                if Delta < 0.1 then
                  H := 0.1 * H
                else
                  begin
                    if 4 <= Delta then
                      H := 4 * H
                    else
                      H := Delta * H;
                  end;
                if H > Hmax then
                  H := Hmax;
                if H < Hmin then
                  H := Hmin;
              end;
          end;
        if (Big < ABS(Y[J])) or (MaxM = J) then
          goto 999;
      end;
999:
    Mend := J;
    M := J;
    if (B > T[J]) or (Big < ABS(T[J])) then
      M := J + 1;
  end;

  procedure MESSAGE (var Step: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN('                       RUNGE-KUTTA-FEHLBERG METHOD');
    WRITELN;
    WRITELN('The Runge-Kutta-Fehlberg method is used to solve the differential equation.');
    WRITELN;
    WRITELN('          y` = f(t,y)  with the initial condition  y(a) = y .');
    WRITELN('                                                           0');
    WRITELN('The solution is computed over  [a,b],  and the step size is automatically');
    WRITELN;
    WRITELN('determined.  Step size is changed to a larger value when the local error');
    WRITELN;
    WRITELN('is small and it is changed to a smaller value if the local error is large.');
    WRITELN;
    WRITELN;
    WRITELN('                       Choose the type of steps used: ');
    WRITELN;
    WRITELN('         <1>  {h }  =  { 1, 0.5, 0.2, 0.1, 0.05, 0.02, 0.01, ...} ');
    WRITELN('                k ');
    WRITELN;
    WRITELN('         <2>  {h }  =  { 1, 1/2, 1/4, 1/8, 1/16, 1/32, 1,64, ...} ');
    WRITELN('                k ');
    WRITELN;
    WRITELN('         <3>   h    is chosen by the computer to be "optimal". ');
    WRITELN('                k ');
    WRITELN;
    Mess := '                       SELECT  < 1 - 3 >  ';
    WRITELN;
    WRITE(Mess);
    READLN(Step);
    if (Step < 1) and (State <> Changes) then
      Step := 1;
    if (Step > 3) and (State <> Changes) then
      Step := 3;
  end;

  procedure GETFUN (var FunType: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('       Choose the D.E. you wish to solve:');
    WRITELN;
    for K := 1 to FunMax do
      begin
        WRITE('                  <', K : 1, '>  ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    Mess := '                       SELECT < 1 - 9 > ?  ';
    WRITELN;
    WRITE(Mess);
    READLN(FunType);
    if (FunType < 1) and (State <> Changes) then
      FunType := 1;
    if (FunType > FunMax) and (State <> Changes) then
      FunType := FunMax;
    CLRSCR;
    WRITELN;
    WRITELN('          The Runge-Kutta-Fehlberg method is used to solve the D.E.');
    WRITELN;
    WRITE('                        ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('          with the initial condition  Y(A) = Y .  A numerical');
    WRITELN('                                              0');
    WRITELN;
    WRITELN('          approximation is computed over  [A,B].  You must enter');
    WRITELN;
    WRITELN;
    WRITELN('          the endpoints for the interval, and the initial condition.');
    WRITELN;
    WRITELN;
    WRITELN;
    WRITE('                        Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
  end;

  procedure EPOINTS (var A, B, Y0: real; var State: STATES);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      I: integer;
      Valu: real;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    if State = Changes then
      Stat := Change;
    while (Stat = Enter) or (Stat = Change) do
      begin
        CLRSCR;
        WRITELN;
        WRITE('                 ');
        PRINTFUNCTION(FunType);
        WRITELN;
        WRITELN;
        if (Stat = Enter) then
          begin
            Mess := '     ENTER  the  left  endpoint  A = ';
            WRITELN;
            WRITE(Mess);
            READLN(A);
            WRITELN;
            Mess := '     ENTER  the  right endpoint  B = ';
            WRITELN;
            WRITE(Mess);
            READLN(B);
            WRITELN;
            Mess := '     ENTER initial condition  Y(A) = ';
            WRITELN;
            WRITE(Mess);
            READLN(Y0);
            WRITELN;
            Mess := '     ENTER  initial  step  size  H = ';
            WRITELN;
            WRITE(Mess);
            READLN(Hstart);
            WRITELN;
          end
        else
          begin
            WRITELN('     The  left  endpoint  is     A =', A : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     The  right endpoint  is     B =', B : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     Initial   condition  is  Y(A) =', Y0 : 15 : 7);
            WRITELN;
            WRITELN;
            WRITELN('     Initial   step size  is     H =', Hstart : 15 : 7);
          end;
        WRITELN;
        WRITELN;
        WRITE('     Want to make a change ?  <Y/N>  ');
        READLN(Resp);
        WRITELN;
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            CLRSCR;
            WRITELN;
            WRITE('                 ');
            PRINTFUNCTION(FunType);
            WRITELN;
            WRITELN('     The current left  endpoint is A =', A : 15 : 7);
            Mess := '     ENTER  the NEW left  endpoint A = ';
            WRITELN;
            WRITE(Mess);
            READLN(A);
            WRITELN;
            WRITELN('     The current right endpoint is B =', B : 15 : 7);
            Mess := '     ENTER  the NEW right endpoint B = ';
            WRITELN;
            WRITE(Mess);
            READLN(B);
            WRITELN;
            WRITELN('     The   current   I. C.  is  Y(A) =', Y0 : 15 : 7);
            Mess := '     Now  ENTER the NEW  I. C.  Y(A) = ';
            WRITELN;
            WRITE(Mess);
            READLN(Y0);
            WRITELN;
            WRITELN('     The initial step size  is     H =', Hstart : 15 : 7);
            Mess := '     ENTER  the  NEW  step  size   H = ';
            WRITELN;
            WRITE(Mess);
            READLN(Hstart);
            WRITELN;
          end
        else
          Stat := Done;
      end;
  end;

  procedure RESULTS (T, Y: VECTOR; M, Mend: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('The Runge-Kutta-Fehlberg was used to compute the solution to');
    WRITELN;
    WRITE('      ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('with  Y(', T[0] : 15 : 7, '  ) =', Y[0] : 15 : 7);
    WRITELN;
    WRITELN('    k           t                   y  ');
    WRITELN('                 k                   k ');
    WRITELN('  ------------------------------------------------');
    WRITELN;
    for K := 0 to Mend do
      begin
        WRITELN(K : 5, '   ', T[K] : 15 : 7, '     ', Y[K] : 15 : 7);
        WRITELN;
        if K mod 11 = 9 then
          begin
            WRITE('                  Press the <ENTER> key.  ');
            READLN(Ans);
            WRITELN;
            WRITELN;
          end;
      end;
    if Mend < M then
      begin
        WRITELN('The solution points are approaching a pole.');
        WRITELN;
      end;
    WRITE('                  Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
  end;

begin                                            {Begin Main Program}
  A := 0;
  B := 1;
  Hstart := 1 / 2;
  Y0 := 0;
  Step := 1;
  FunType := 1;
  State := Working;
  while Step <> 0 do
    begin
      MESSAGE(Step);
      DoMo := Go;
      while DoMo = Go do
        begin
          GETFUN(FunType);
          while (State = Working) or (State = Changes) do
            begin
              EPOINTS(A, B, Y0, State);
              RKF45(A, B, Y0, Tol, TolLo, TolHi, Step, T, Y, M, Mend);
              RESULTS(T, Y, M, Mend);
              WRITELN;
              WRITELN;
              WRITE('Do you want to try another  initial condition ?  <Y/N>  ');
              READLN(Ans);
              WRITELN;
              if (Ans <> 'Y') and (Ans <> 'y') then
                State := Done
              else
                State := Changes;
            end;
          WRITELN;
          WRITE('Want  to  change  the  differential  equation ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            DoMo := Stop
          else
            State := Changes;
        end;
      WRITELN;
      WRITE('Want to try a  different  step size  strategy ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Step := 0
      else
        State := Changes;
    end;
end.                                            {End of Main Program}

